k6 v1.6.0 is here 🎉! This release includes:

- Cloud commands now support configurable default Grafana Cloud stack.
- New `k6 deps` command for analyzing script dependencies.
- Browser APIs enhancements with `frameLocator()`, `goBack()`, `goForward()` methods.
- Crypto module adds PBKDF2 support for password-based key derivation.
- `jslib` gets a new TOTP library for time-based one-time password generation and validation.
- New [mcp-k6](https://github.com/grafana/mcp-k6) MCP server for AI-assisted k6 script writing.

## Breaking changes

There are no breaking changes in this release.

## New features

### Configurable default stack for Cloud commands [#5420](https://github.com/grafana/k6/pull/5420)

Cloud commands now support configuring the default Grafana Cloud stack you want to use. The stack slug (or stack id) is used by the Cloud to determine which default project to use when not explicitly provided.

Previously, users had to specify the project id for every test run. With this change, you can configure a default stack during login, and k6 will use it to automatically resolve the appropriate default project. This is particularly useful for organizations with multiple Grafana Cloud stacks or when working across different teams and environments.

Users can also set up a specific stack for every test run, either using the new option `stackID` or the environment variable `K6_CLOUD_STACK_ID`.

Please note that, in k6 v2, this stack information will become **mandatory** to run a test.

```bash
# Login interactively and select default stack
k6 cloud login

# Login and set default stack with token
k6 cloud login --token $MY_TOKEN --stack my-stack-slug

# Run test using the configured default stack
k6 cloud run script.js

# Run test using a specific stack
K6_CLOUD_STACK_ID=12345 k6 cloud run script.js

# Stack id can also be set in the options
export const options = {
  cloud: {
    stackID: 123,
    projectID: 789,  // If the project does not belong to the stack, this will throw an error
  },
};
```

This simplifies the cloud testing workflow and prepares k6 for upcoming changes to the Grafana Cloud k6 authentication process, where the stack will eventually become mandatory.

### `k6 deps` command and manifest support [#5410](https://github.com/grafana/k6/pull/5410), [#5427](https://github.com/grafana/k6/pull/5427)

A new `k6 deps` command is now available to analyze and list all dependencies of a given script or archive. This is particularly useful for understanding which extensions are required to run a script, especially when using auto extension resolution.

The command identifies all imports in your script and lists dependencies that might be needed for building a new binary with auto extension resolution. Like auto extension resolution itself, this only accounts for imports, not dynamic `require()` calls.

```bash
# Analyze script dependencies
k6 deps script.js

# Output in JSON format for programmatic consumption
k6 deps --json script.js

# Analyze archived test dependencies
k6 deps archive.tar
```

This makes it easier to understand extension requirements, share scripts with clear dependency information, and integrate k6 into automated build pipelines.

In addition, k6 now supports a manifest that specifies default version constraints for dependencies when no version is defined in the script using pragmas. If a dependency is imported without an explicit version, it defaults to "*", and the manifest can be used to replace that with a concrete version constraint.

The manifest is set through an environment variable as JSON with keys being a dependency and values being constraints:

```
K6_DEPENDENCIES_MANIFEST='{"k6/x/faker": ">=v0.4.4"}' k6 run scripts.js
```

In this example, if the script only imports `k6/x/faker` and does not use a `use k6 with k6/x/faker ...` directive, it will set the version constraint to `>=v0.4.4`. It will not make any changes if `k6/x/faker` is not a dependency of the script at all.

### Browser module: `frameLocator()` method [#5487](https://github.com/grafana/k6/pull/5487)

The browser module now supports [`frameLocator()`](https://grafana.com/docs/k6/latest/javascript-api/k6-browser/locator/framelocator/) on `Page`, `Frame`, `Locator`, and `FrameLocator` objects. This method creates a locator for working with `iframe` elements without the need to explicitly switch contexts, making it much easier to interact with embedded content.

Frame locators are particularly valuable when testing applications with nested iframes, as they allow you to chain locators naturally while maintaining readability:

<details>
<summary>Click to expand example code</summary>

```javascript
import { browser } from 'k6/browser';

export const options = {
  scenarios: {
    ui: {
      executor: 'shared-iterations',
      options: {
        browser: {
          type: 'chromium',
        },
      },
    },
  },
};

export default async function () {
  const page = await browser.newPage();

  try {
    await page.goto('https://example.com');

    // Locate an iframe and interact with elements inside it
    const frame = page.frameLocator('#payment-iframe');
    await frame.locator('#card-number').fill('4242424242424242');
    await frame.locator('#submit-button').click();

    // Chain frame locators for nested iframes
    const nestedFrame = page
      .frameLocator('#outer-frame')
      .frameLocator('#inner-frame');
    await nestedFrame.locator('#nested-content').click();
  } finally {
    await page.close();
  }
}
```

</details>

This complements existing frame handling methods and provides a more intuitive API for working with iframe-heavy applications.

### Browser module: `goBack()` and `goForward()` navigation methods [#5494](https://github.com/grafana/k6/pull/5494)

The browser module now supports [`page.goBack()`](https://grafana.com/docs/k6/latest/javascript-api/k6-browser/page/goback/) and [`page.goForward()`](https://grafana.com/docs/k6/latest/javascript-api/k6-browser/page/goforward/) methods for browser history navigation. These methods allow you to navigate the page's history, similar to clicking the browser's back/forward buttons.

<details>
<summary>Click to expand example code</summary>

```javascript
import { browser } from 'k6/browser';

export const options = {
  scenarios: {
    ui: {
      executor: 'shared-iterations',
      options: {
        browser: {
          type: 'chromium',
        },
      },
    },
  },
};

export default async function () {
  const page = await browser.newPage();

  try {
    await page.goto('https://example.com');
    await page.goto('https://example.com/page2');
    
    // Navigate back to the previous page
    await page.goBack();
    
    // Navigate forward again
    await page.goForward();
    
    // Both methods support optional timeout and waitUntil parameters
    await page.goBack({ waitUntil: 'networkidle' });
  } finally {
    await page.close();
  }
}
```

</details>

### Browser module: Request event handlers [#5481](https://github.com/grafana/k6/pull/5481), [#5486](https://github.com/grafana/k6/pull/5486)

The browser module now supports `page.on('requestfailed')` and `page.on('requestfinished')` event handlers, enabling better monitoring and debugging of network activity during browser tests.

The `requestfailed` event fires when a request fails (network errors, aborts, etc.), while `requestfinished` fires when a request completes successfully. 

<details>
<summary>Click to expand example code</summary>

```javascript
import { browser } from 'k6/browser';

export const options = {
  scenarios: {
    ui: {
      executor: 'shared-iterations',
      options: {
        browser: {
          type: 'chromium',
        },
      },
    },
  },
};

export default async function () {
  const page = await browser.newPage();

  // Monitor failed requests
  page.on('requestfailed', (request) => {
    console.log(`Request failed: ${request.url()}`);
  });

  // Monitor successful requests
  page.on('requestfinished', (request) => {
    console.log(`Request finished: ${request.url()}`);
  });

  await page.goto('https://example.com');
  await page.close();
}
```

</details>

These event handlers provide deeper insights into network behavior during browser testing and help identify issues that might not be immediately visible.

### Crypto module: PBKDF2 support [#5380](https://github.com/grafana/k6/pull/5380)

The crypto module now supports PBKDF2 for deriving cryptographic keys from passwords. PBKDF2 is widely used for password hashing and key derivation in security-sensitive applications, and this addition enables testing of systems that use PBKDF2 for authentication or encryption.

For usage examples, check out the [one provided](https://github.com/grafana/k6/blob/b7c19129a8dc5a69bb519e7b313eab6473114e6f/examples/webcrypto/derive_key/derive-key-pbkdf2.js) in the repository or refer to the [documentation](https://grafana.com/docs/k6/latest/javascript-api/crypto/pbkdf2params).

### WebSockets module is now stable [#5586](https://github.com/grafana/k6/pull/5586)

The websockets module has been promoted to stable status and is now available via the `k6/websockets` path.
The experimental `k6/experimental/websockets` module will be removed in a future release. Users should migrate to the stable `k6/websockets` module.

To migrate, simply update your import statement:

```javascript
// Old (experimental)
import ws from 'k6/experimental/websockets';

// New (stable)
import ws from 'k6/websockets';
```

No other changes are required because the API is the same.

### Console logging: ArrayBuffer and TypedArray support [#5496](https://github.com/grafana/k6/pull/5496)

`console.log()` now properly displays `ArrayBuffer` and `TypedArray` objects, making it easier to debug binary data handling in your test scripts. Previously, these types would not display useful information, making debugging difficult when working with binary protocols, file uploads, or WebSocket binary messages.

<details>
<summary>Click to expand example code</summary>

```javascript
// Log ArrayBuffer - shows detailed byte contents
const buffer = new ArrayBuffer(8);
const view = new Int32Array(buffer);
view[0] = 4;
view[1] = 2;
console.log(buffer);
// Output: ArrayBuffer { [Uint8Contents]: <04 00 00 00 02 00 00 00>, byteLength: 8 }

// Log TypedArrays - shows type, length, and values
const int32 = new Int32Array([4, 2]);
console.log(int32);
// Output: Int32Array(2) [ 4, 2 ]

// Nested objects with TypedArrays
console.log({ v: int32 });
// Output: { v: Int32Array(2) [ 4, 2 ] }

// Complex nested structures
console.log({ 
  name: "test", 
  buffer: buffer, 
  view: int32 
});
// Output: { name: "test", buffer: ArrayBuffer {...}, view: Int32Array(2) [...] }
```

</details>

### Configurable TLS version for Experimental Prometheus output [#5537](https://github.com/grafana/k6/pull/5537)

The experimental Prometheus remote write output now supports configuring the minimum TLS version used for connections. This allows you to meet specific security requirements or compatibility constraints when sending metrics to Prometheus endpoints.

If not set, the default minimum TLS version is 1.3.

```bash
K6_PROMETHEUS_RW_TLS_MIN_VERSION=1.3 k6 run script.js -o experimental-prometheus-rw
```

### A new TOTP library [k6-totp](https://github.com/grafana/k6-jslib-totp)

A new TOTP (Time-based One-Time Password) library is now available in [jslib.k6.io](https://jslib.k6.io), enabling k6 scripts to generate and validate time-based one-time passwords. This is particularly useful for testing applications that use TOTP-based two-factor authentication (2FA), such as authenticator apps like Google Authenticator or Authy. See the documentation at [k6-totp](https://grafana.com/docs/k6/latest/javascript-api/jslib/totp/).

<details>
<summary>Click to expand example code</summary>

```javascript
import http from 'k6/http';
import { TOTP } from 'https://jslib.k6.io/totp/1.0.0/index.js';

export default async function () {
  // Initialize TOTP with your secret key (base32 encoded)
  // The second parameter is the number of digits (typically 6 or 8)
  const totp = new TOTP('GEZDGNBVGY3TQOJQGEZDGNBVGY3TQOJQ', 6);

  // Generate the current TOTP code
  const code = await totp.gen();
  console.log(`Generated TOTP code: ${code}`);

  // Use the TOTP code in authentication
  const response = http.post('https://api.example.com/login', JSON.stringify({
    username: 'user@example.com',
    password: 'password123',
    totpCode: code
  }), {
    headers: { 'Content-Type': 'application/json' }
  });

  // Optionally verify a TOTP code
  const isValid = await totp.verify(code);
  console.log(`Code is valid: ${isValid}`);
}
```

</details>

The library follows [the standard TOTP RFC 6238 specification](https://datatracker.ietf.org/doc/html/rfc6238), ensuring compatibility with standard authenticator applications.

### Introducing mcp-k6: AI-assisted k6 script writing [mcp-k6](https://github.com/grafana/mcp-k6)

[mcp-k6](https://github.com/grafana/mcp-k6) is an experimental [Model Context Protocol](https://modelcontextprotocol.io/) (MCP) server for k6. Once connected to your AI assistant or MCP-compatible editor (such as Cursor, VS Code, or Claude Desktop), it helps you write better k6 scripts faster and run them with confidence.

With mcp-k6, your AI assistant can:

- **Write accurate scripts:** Create up-to-date scripts by referring to embedded k6 documentation and TypeScript definitions to reduce API hallucinations.
- **Validate scripts:** Catch syntax errors, missing imports, and `export default function` declarations before execution.
- **Run tests locally:** Execute scripts and review results without leaving your editor.
- **Generate scripts:** Create tests from requirements using guided prompts that follow k6 best practices.
- **Convert browser tests:** Transform Playwright tests into k6 browser scripts while preserving test logic.

To get started, install mcp-k6 via Docker, Homebrew, or from source, then configure it with your editor. See the [documentation](https://grafana.com/docs/k6/latest/set-up/configure-ai-assistant/) for installation instructions and setup guides.

## UX improvements and enhancements

- [#5423](https://github.com/grafana/k6/pull/5423) Adds new execution status `ExecutionStatusMarkedAsFailed` when a test is explicitly marked as failed using `exec.test.fail()`.
- [#5590](https://github.com/grafana/k6/pull/5590) Adds deprecation notice to encourage users to configure their stack when using Cloud CLI commands.
- [#5607](https://github.com/grafana/k6/pull/5607), [#5608](https://github.com/grafana/k6/pull/5608) Documentation improvements for Cloud CLI commands.

## Bug fixes

- [#5503](https://github.com/grafana/k6/pull/5503) Fixes a memory leak in `csv.parse()` where each VU created its own copy of CSV data instead of sharing it.
- [#5425](https://github.com/grafana/k6/pull/5425) Fixes CLI validation of summary trend stats to properly accept valid configurations.
- [#5132](https://github.com/grafana/k6/pull/5132) Fixes missing exception when using `saltLength: 0` with RSA-PSS sign/verify operations.
- [#5602](https://github.com/grafana/k6/pull/5602) Fixes `Config.String()` in the OpenTelemetry output to properly display `ExporterProtocol`.
- [#5578](https://github.com/grafana/k6/pull/5578) Fixes error hiding when the secret value is empty.
- [#5546](https://github.com/grafana/k6/pull/5546) Corrects JSON error position reporting in HTTP requests.
- [#5526](https://github.com/grafana/k6/pull/5526) Fixes browser context cancellation during target attachment to prevent test flakiness.
- [#5611](https://github.com/grafana/k6/pull/5611) Fixes the fallback directory for the binaries cache when it's not possible to get one from the system.

## Maintenance and internal improvements

- [#5603](https://github.com/grafana/k6/pull/5603) Upgrades Alpine images in the Dockerfile to the latest versions.
- [#5498](https://github.com/grafana/k6/pull/5498) Makes `WaitForEvent` mapping more type-safe and consistent by using a typed event name parameter instead of a raw string.
- [#5561](https://github.com/grafana/k6/pull/5561) Improves context-cancellation visibility in browser module by preserving and surfacing deterministic cancellation causes, helping diagnose race conditions and flaky tests.
- [#5414](https://github.com/grafana/k6/pull/5414) Makes `make generate` command OS-agnostic for cross-platform compatibility.
- [#5245](https://github.com/grafana/k6/pull/5245) Replaces `github.com/jhump/protoreflect/desc/protoparse` with `github.com/bufbuild/protocompile` for better protobuf parsing.
- [#5465](https://github.com/grafana/k6/pull/5465) Adds new `cloudapi/v6` package with authentication features.
- [#5573](https://github.com/grafana/k6/pull/5573) Optimizes cloud login by running legacy migration only once.
- [#5570](https://github.com/grafana/k6/pull/5570) Moves browser tests into its own test workflow for improved CI performance.
- [#5560](https://github.com/grafana/k6/pull/5560) Refactors browser module mouse options parsing to mapping layer.
- [#5554](https://github.com/grafana/k6/pull/5554) Refactors `onAttachedToTarget` context done detection for better maintainability.
- [#5553](https://github.com/grafana/k6/pull/5553) Fixes `TestFlushMaxSeriesInBatch` non-deterministic behavior.
- [#5489](https://github.com/grafana/k6/pull/5489) Addresses `frameLocator` PR review comments and improves code organization.
- [#5559](https://github.com/grafana/k6/pull/5559) Updates issue auto-assignment to exclude internal team members.
- [#5536](https://github.com/grafana/k6/pull/5536) Configures stale PR workflow to close PRs after 2 months of inactivity.
- [#5568](https://github.com/grafana/k6/pull/5568) Disables flaky tests in CI.
- [#5531](https://github.com/grafana/k6/pull/5531) Improves release template for TypeScript fork synchronization.
- [#5509](https://github.com/grafana/k6/pull/5509), [#5508](https://github.com/grafana/k6/pull/5508) Updates error logging from error to debug level for user-unactionable internal issues.
- [#5621](https://github.com/grafana/k6/pull/5621) Disables `t.Parallel()` on `TestCompile` for now, to reduce flakiness in CI.

## Roadmap

We've started looking into k6 v2! The major release will focus on introducing breaking changes that have accumulated throughout the v1.x series, such as removing deprecated APIs and changing default behaviors. New features will continue to be incrementally released in v1.x minor versions as usual until the v2.0.0 release.

## External contributors

A huge thank you to the external contributors who helped during this release: @LBaronceli, @baeseokjae, @chojs23, @pkalsi97, @weillercarvalho, @ariasmn, @rahulmedicharla, @janHildebrandt98 and @shota3506! 🙏
